<?php
/**
 * @version     1.0.0
 * @package     K2 to Content Migration
 * @copyright   Copyright (C) 2023 JoomHelper.com. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

class K2toContentMigrationController extends JControllerLegacy
{
    protected $categoryMap = []; //[old_id] => [new_id]

    protected $fieldGroupMap = [];

    protected $fieldMap = [];

    protected $fieldMapType = [];

    protected $assetId = 0;

    protected $code = 0;

    protected $limit  = 30;

    protected $offset = 0;

    protected $total  = 0;

    protected $fileAttachmentPlugin = JPATH_ROOT.'/plugins/content/jmpattachments/jmpattachments.php';

    public function display($cachable = false, $urlparams = [])
    {
        $app = JFactory::getApplication();
        $doc = JFactory::getDocument();
        
        JToolBarHelper::title("K2 to Joomla Content");

        // check k2 installed
        $k2 = JPATH_ROOT."/components/com_k2/k2.php";
        if(!file_exists($k2))
        {
            ?>
            <p>Opps! The K2 extension has not been installed on your website.</p>
            <?php
            return;
        }

        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        // count articles
        $query->clear()
              ->select("COUNT(*)")
              ->from("#__k2_items")
              ->where("trash=0");

        $articleCount = (int) $db->setQuery($query)->loadResult();

        $query->clear()
              ->select("COUNT(*)")
              ->from("#__k2_categories")
              ->where("trash=0");

        $catCount = (int) $db->setQuery($query)->loadResult();

        ?>
        <div class="row">
            <div class="span9" style="text-align:center; font-size:1.1em">
                <p>It is estimated that <strong><?=$articleCount?></strong> articles and <strong><?=$catCount?></strong> categories will be imported into Joomla Content.</p>
                <p style="color:red; font-weight:bold">IMPORTANT: Alway backup your site FIRST.</p>
                <hr>
                
                <div style="text-align: left; padding-left: 50px;">
                    <table class="table">
                        <tbody>
                            <tr>
                                <td width="200px">
                                    <label for="introimage">Process K2 Intro Image</label>
                                </td>
                                <td>
                                    <select id="introimage" style="margin:0 5px">
                                        <option value="">None</option>
                                        <option value="1" selected>Use for Intro Image</option>
                                        <option value="2">Use for Full Article Image</option>
                                        <option value="3">Both</option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <td>
                                    <label for="keepid" style="display:inline">Keep K2 ID</label>
                                </td>
                                <td>
                                    <input type="checkbox" id="keepid" style="margin:0 0 0 5px;">
                                </td>
                            </tr>
                            <tr>
                                <td>
                                    <label for="attachment" style="display:inline">K2 Attachment<br>
                                        <small>Requires <a href="https://joomhelper.com/n9ejyYy" target="_blank">File Attachment</a> plugin.</small>
                                    </label>
                                </td>
                                <td><input type="checkbox" id="attachment" <?php echo !file_exists($this->fileAttachmentPlugin)?'disabled':'' ?> style="margin:0 0 0 5px;"></td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <hr>
                <button id="k2ImportBtn" type="button" class="btn btn-large btn-primary">Import Now</button>

                <div id="k2Result" class="hidden hide">
                    <div>Success: <span class="__1">0</span></div>
                    <div>Failure: <span class="__0">0</span></div>

                    <div class="log" style="border: 1px dashed red; margin: 20px; padding: 10px; border-radius: 5px; color: red; white-space: pre; text-align: left;">
                    </div>
                </div>
            </div>

            <div class="span3">
                <iframe src="about:blank" data-url="https://joomhelper.com/?view=frame" style="border: 1px solid #b8c8e0; background: #fff; width: 100%; height: 500px; border-radius: 5px; margin: 0 0 20px 0; position: sticky; top: 100px;"></iframe>
            </div>
        </div>
        <?php

        JHtml::_("behavior.keepalive");
        JHtml::_("jquery.framework");

        $doc->addScript(JUri::base(true)."/components/com_k2tocontentmigration/script.js?v=1.3");
        $doc->addScriptDeclaration(';window.baseUrl="'.JUri::base(true).'"');
    }

    protected function getCoreArticles()
    {
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        $query->select("SQL_CALC_FOUND_ROWS a.*")
              ->select("c.name catName")
              ->from("#__k2_items a")
              ->leftJoin("#__k2_categories c ON a.catid=c.id")
              ->order("a.id ASC")
              ->where("a.trash=0")
              ->group("a.id");

        $articles = $db->setQuery($query, $this->offset, $this->limit)->loadObjectList();

        $this->total = (int) $db->setQuery("SELECT FOUND_ROWS()")->loadResult();

        $return = [
            "offset" => $this->offset,
            "limit" => $this->limit,
            "total" => $this->total,
            "articles" => $articles
        ];

        return $return;
    }

    public function beginImport()
    {
        set_time_limit(0);

        $app = JFactory::getApplication();
        $code = $app->input->getInt("code", 0);

        if(!preg_match("/^[1-9][0-9]*$/", $code)) return;

        $this->code = $code;

        // create result file
        file_put_contents(dirname(__FILE__)."/".$code.".ini", json_encode(["success"=>0, "failure"=>0]));

        // find assetId of content
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        $query->select("id")
              ->from("#__assets")
              ->where("name=".$db->quote("com_content"));

        $this->assetId = $db->setQuery($query)->loadResult();

        $this->importCategories(0, 1, $this->assetId, 1);
        $this->importFields();
        $this->importContent();

        // rebuild category
        try {
            JTable::addIncludePath(JPATH_ADMINISTRATOR."/components/com_categories/tables");
            $table = JTable::getInstance("Category", "CategoriesTable", []);
            $table->rebuild();
        } catch(Exception $e){};
    }

    protected function importContent()
    {
        $resultFile = dirname(__FILE__)."/".$this->code.".ini";
        $logFile = dirname(__FILE__)."/".$this->code."_log.txt";
        file_put_contents($logFile, "--- LOG FILE ---\n");

        $data = $this->getCoreArticles();

        if(empty($data["articles"]))
        {
            // done
            $result = json_decode(file_get_contents($resultFile), true);
            $result["done"] = 1;

            file_put_contents($resultFile, json_encode($result));

            return;
        }

        $app = JFactory::getApplication();
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);
        $introImg = $app->input->getInt("intro", 0);
        $keepid = $app->input->getInt("keepid", 0);
        $attachment = $app->input->getInt("attachment", 0);

        if($attachment && !file_exists($this->fileAttachmentPlugin)) $attachment=0;

        if($attachment)
        {
            $query->clear()
                  ->select('params')
                  ->from('#__extensions')
                  ->where([
                    'element='.$db->quote('jmpattachments'),
                    'type='.$db->quote('plugin')
                  ]);

            $jparams = $db->setQuery($query)->loadResult();

            if(!$jparams)
            {
                $attachment = 0;
            } else {
                $jparams = json_decode($jparams);
                $pathFolder = isset($jparams->folder)?$jparams->folder:'media/jattachments';

                jimport('joomla.filesystem.file');
                jimport('joomla.filesystem.folder');

                if(!JFolder::exists(JPATH_ROOT.'/'.$pathFolder) && !JFolder::create(JPATH_ROOT.'/'.$pathFolder, 755))
                {
                    $attachment = 0;
                }
            }
        }

        foreach($data["articles"] as $a)
        {
            $row = [
                "title" => $a->title,
                "alias" => $a->alias,
                "introtext" => $a->introtext,
                "fulltext" => $a->fulltext,
                "state" => $a->published,
                "created" => $a->created,
                "created_by" => $a->created_by,
                "created_by_alias" => $a->created_by_alias,
                "modified" => $a->modified,
                "modified_by" => $a->modified_by,
                "publish_up" => $a->publish_up,
                "publish_down" => $a->publish_down,
                "metakey" => $a->metakey,
                "metadesc" => $a->metadesc,
                "access" => $a->access,
                "hits" => $a->hits,
                "metadata" => $a->metadata,
                "featured" => $a->featured,
                "language" => $a->language,
            ];

            if($keepid)
            {
                $query->clear()
                      ->select("COUNT(*)")
                      ->from("#__content")
                      ->where("id=".$db->quote($a->id));
                $ex = $db->setQuery($query)->loadResult();

                if(!$ex) $row["id"]=$a->id;
            }

            // alias
            $query->clear()
                  ->select("COUNT(*)")
                  ->from("#__content")
                  ->where("alias=".$db->quote($a->alias));

            $exist = $db->setQuery($query)->loadResult();

            if($exist) $a->alias.="-".time();
            $row["alias"] = $a->alias;

            // category
            $row["catid"] = array_key_exists($a->catid, $this->categoryMap)?$this->categoryMap[$a->catid]:1;

            // images
            $image = JPATH_ROOT."/media/k2/items/src/".md5("Image".$a->id).".jpg";
            if(file_exists($image))
            {
                $dtImg = [];

                if($introImg==1||$introImg==3)
                {
                    $dtImg["image_intro"]="images/k2/".md5("Image".$a->id).".jpg";
                    $dtImg["image_intro_caption"]="";
                    $dtImg["image_intro_alt"]="";
                }

                if($introImg==2||$introImg==3)
                {
                    $dtImg["image_fulltext"]="images/k2/".md5("Image".$a->id).".jpg";
                    $dtImg["image_fulltext_caption"]="";
                    $dtImg["image_fulltext_alt"]="";
                }

                $row["images"]=json_encode($dtImg);

                JFolder::create(JPATH_ROOT."/images/k2");
                JFile::copy($image, JPATH_ROOT."/images/k2/".md5("Image".$a->id).".jpg");
            }

            // import
            $columns = [];
            $values = [];

            foreach($row as $k=>$v)
            {
                $columns[]=$k;
                $values[]=$db->quote($v);
            }

            $query->clear()
                  ->insert($db->quoteName("#__content"))
                  ->columns($db->quoteName($columns))
                  ->values(implode(',', $values));

            $result = json_decode(file_get_contents($resultFile));

            try {
                $db->setQuery($query)->execute();

                // new article ID
                $new_row_id = $db->insertid();

                // ucm_content
                $query->clear()
                      ->insert('#__ucm_content')
                      ->columns($db->quoteName([
                        'core_type_alias',
                        'core_title',
                        'core_alias',
                        'core_body',
                        'core_state',
                        'core_access',
                        'core_params',
                        'core_created_user_id',
                        'core_created_time',
                        'core_images',
                        'core_urls',
                        'core_hits',
                        'core_version',
                        'core_catid',
                        'core_type_id',
                        'core_modified_time',
                        'core_content_item_id'
                      ]))
                      ->values(implode(',', [
                        $db->quote('com_content.article'),
                        $db->quote($row["title"]),
                        $db->quote($row["alias"]),
                        $db->quote($row["introtext"]),
                        1,
                        1,
                        $db->quote($a->attribs),
                        $db->quote($a->created_by),
                        $db->quote($a->created),
                        $db->quote($a->images),
                        $db->quote($a->urls),
                        $a->hits,
                        1,
                        $db->quote($row["catid"]),
                        1,
                        $db->quote('0000-00-00 00:00:00'),
                        $new_row_id
                      ]));
                $db->setQuery($query)->execute();

                // new core article ID
                $new_core_row_id = $db->insertid();

                // import field
                if(!empty($a->extra_fields))
                {
                    $fields = json_decode($a->extra_fields);
                    if(!empty($fields))
                    {
                        foreach($fields as $field)
                        {
                            $this->importField($field, $new_row_id);
                        }
                    }
                }

                // import tags
                $this->importTags($new_row_id, $a->id, $new_core_row_id);

                // import attachments.
                if($attachment)
                {
                    $query->clear()
                          ->select('*')
                          ->from('#__k2_attachments')
                          ->where('itemID='.$db->quote($a->id));

                    $listFile = $db->setQuery($query)->loadObjectList();

                    if($listFile)
                    {
                        $this->importFiles($listFile, $pathFolder, $new_row_id, $a);
                    }
                }

                $result->success+=1;
            } catch (Exception $e) {
                $result->failure+=1;

                // add to log file
                $error = "- Item: ".$a->title.":".$e->getMessage()."\n";

                file_put_contents($logFile, $error, FILE_APPEND | LOCK_EX);
            }

            file_put_contents($resultFile, json_encode($result));
        }

        // continue;
        $this->offset+=$this->limit;
        $this->importContent();
    }

    public function readImportResult()
    {
        $app = JFactory::getApplication();
        $code = $app->input->getInt("code", "");

        if(!preg_match("/^[1-9][0-9]*$/", $code)) $this->toJson(["status"=>400]);

        $result_file = dirname(__FILE__)."/".$code.".ini";

        if(!file_exists($result_file))
            $this->toJson(["status"=>400]);

        $result = json_decode(file_get_contents($result_file));
        if(!$result)
            $this->toJson(["status"=>400]);

        $this->toJson($result);
    }

    protected function importCategories($parent=0, $new_parent_id=1, $new_asset_id=0, $level=1)
    {
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        $user = JFactory::getUser();
        $date = JFactory::getDate()->toSql();

        $query->select("*")
              ->from("#__k2_categories")
              ->where([
                "trash=0",
                "parent=".$db->quote($parent)
              ]);

        $list = $db->setQuery($query)->loadObjectList();

        if(!$list) return;

        foreach($list as $cat)
        {
            // import category
            $data = [
                "level" => 1,
                "parent_id" => $new_parent_id,
                "extension" => "com_content",
                "title" => $cat->name,
                "description" => $cat->description,
                "published" => $cat->published,
                "access" => $cat->access,
                "language" => $cat->language,

                "checked_out" => 0,
                "checked_out_time" => "0000-00-00 00:00:00",
                "created_user_id" => $user->id,
                "created_time" => $date
            ];

            if(!empty($cat->image) && $cat->image!='')
            {
                $data['params'] = json_encode([
                    'image' => 'media/k2/categories/'.$cat->image,
                    'image_alt' => ''
                ]);
            }

            // check alias
            $query->clear()
                  ->select("COUNT(*)")
                  ->from("#__categories")
                  ->where("alias=".$db->quote($cat->alias));

            $exist = $db->setQuery($query)->loadResult();

            if($exist) $cat->alias.="-".time();

            $data["alias"] = $cat->alias;
            $data["path"] = $cat->alias;

            $columns = [];
            $values = [];

            foreach($data as $k=>$v)
            {
                $columns[]=$k;
                $values[]=$db->quote($v);
            }

            $query->clear()
                  ->insert($db->quoteName("#__categories"))
                  ->columns($db->quoteName($columns))
                  ->values(implode(',', $values));

            $db->setQuery($query)->execute();

            $new_row_id = $db->insertid();

            // create assets
            if($new_asset_id>0)
            {
                // find max lft
                try {
                    // delete duplicate row
                    $query->clear()
                          ->delete('#__assets')
                          ->where('name='.$db->quote('com_content.category.'.$new_row_id));
                    $db->setQuery($query)->execute();

                    $query->clear()
                          ->select("MAX(lft)")
                          ->from("#__assets");
                    $max = $db->setQuery($query)->loadResult();
                    $max = !$max?0:(int)$max;

                    $query->clear()
                          ->insert($db->quoteName("#__assets"))
                          ->columns($db->quoteName(['parent_id', 'lft', 'rgt', 'level', 'name', 'title', 'rules']))
                          ->values(implode(',', [$new_asset_id, ($max+2), ($max+3), $level, $db->quote('com_content.category.'.$new_row_id), $db->quote($data["title"]), $db->quote('{}')]));

                    $db->setQuery($query)->execute();
                    $new_asset_id = $db->insertid();
                } catch(Exception $e){
                    $new_asset_id = 0;
                };

                if($new_asset_id)
                {
                    $query->clear()
                          ->update("#__categories")
                          ->set("asset_id=".$db->quote($new_asset_id))
                          ->where("id=".$db->quote($new_row_id));

                    $db->setQuery($query)->execute();
                }
            }

            $this->categoryMap[$cat->id]=$new_row_id;

            // import field groups
            if($cat->extraFieldsGroup>0 && array_key_exists($cat->extraFieldsGroup, $this->fieldGroupMap))
            {
                $fieldGroupId = $this->fieldGroupMap[$cat->extraFieldsGroup];
                $query->clear()
                      ->select("id")
                      ->from("#__fields")
                      ->where("group_id=".$db->quote($fieldGroupId))
                      ->order("id ASC");

                $fieldCid = $db->setQuery($query)->loadColumn();

                if($fieldCid)
                {
                    foreach($fieldCid as $fieldId)
                    {
                        $query->clear()
                              ->insert($db->quoteName("#__fields_categories"))
                              ->columns($db->quoteName([
                                    "field_id",
                                    "category_id"
                              ]))
                              ->values(implode(',', [
                                    $fieldId,
                                    $new_row_id
                              ]));

                        $db->setQuery($query)->execute();
                    }
                }
            }

            // import child categories
            $this->importCategories($cat->id, $new_row_id, $new_asset_id, $level+1);
        }
    }

    protected function importCategory($k2CatId, $child=[])
    {
        if(array_key_exists($k2CatId, $this->categoryMap))
            return $this->categoryMap[$k2CatId];

        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        $user = JFactory::getUser();
        $date = JFactory::getDate()->toSql();

        $query->select("*")
              ->from("#__k2_categories")
              ->where([
                "trash=0",
                "id=".$db->quote($k2CatId)
              ]);

        $cat = $db->setQuery($query)->loadObject();
        if(!$cat)
        {
            $this->categoryMap[$k2CatId]=0;
            return 0;
        }

        // import category
        $data = [
            "level" => 1,
            "parent_id" => 1,
            "extension" => "com_content",
            "title" => $cat->name,
            "description" => $cat->description,
            "published" => $cat->published,
            "access" => $cat->access,
            "language" => $cat->language,

            "checked_out" => 0,
            "checked_out_time" => "0000-00-00 00:00:00",
            "created_user_id" => $user->id,
            "created_time" => $date
        ];

        // check alias
        $query->clear()
              ->select("COUNT(*)")
              ->from("#__categories")
              ->where("alias=".$db->quote($cat->alias));

        $exist = $db->setQuery($query)->loadResult();

        if($exist) $cat->alias.="-".time();

        $data["alias"] = $cat->alias;
        $data["path"] = $cat->alias;

        $columns = [];
        $values = [];

        foreach($data as $k=>$v)
        {
            $columns[]=$k;
            $values[]=$db->quote($v);
        }

        $query->clear()
              ->insert($db->quoteName("#__categories"))
              ->columns($db->quoteName($columns))
              ->values(implode(',', $values));

        $db->setQuery($query)->execute();

        $new_row_id = $db->insertid();

        // create assets
        $new_asset_id = 0;
        if($this->assetId>0)
        {
            // find max lft
            $query->clear()
                  ->select("MAX(lft)")
                  ->from("#__assets");
            $max = $db->setQuery($query)->loadResult();
            $max = !$max?0:(int)$max;

            $query->clear()
                  ->insert($db->quoteName("#__assets"))
                  ->columns($db->quoteName(['parent_id', 'lft', 'rgt', 'level', 'name', 'title', 'rules']))
                  ->values(implode(',', [$this->assetId, ($max+2), ($max+3), 1, $db->quote('com_content.category.'.$new_row_id), $db->quote($data["title"]), $db->quote('{}')]));

            $db->setQuery($query)->execute();
            $new_asset_id = $db->insertid();

            if($new_asset_id)
            {
                $query->clear()
                      ->update("#__categories")
                      ->set("asset_id=".$db->quote($new_asset_id))
                      ->where("id=".$db->quote($new_row_id));

                $db->setQuery($query)->execute();

                // update child if have (parent_id, level)
                if(!empty($child))
                {
                    $query->clear()
                          ->select("asset_id")
                          ->from("#__categories")
                          ->where("id IN(".implode(",", $child).")");

                    $assetCid = $db->setQuery($query)->loadColumn();

                    if(!empty($assetCid))
                    {
                        $query->clear()
                              ->update("#__assets")
                              ->set([
                                "level=level+1",
                                "parent_id=IF(parent_id!=".$this->assetId.",parent_id,".$new_asset_id.")"
                              ])
                              ->where("id IN(".implode(",", $assetCid).")");

                        $db->setQuery($query)->execute();
                    }
                }
            }
        }

        $this->categoryMap[$k2CatId]=$new_row_id;

        // import field groups
        if($cat->extraFieldsGroup>0 && array_key_exists($cat->extraFieldsGroup, $this->fieldGroupMap))
        {
            $fieldGroupId = $this->fieldGroupMap[$cat->extraFieldsGroup];
            $query->clear()
                  ->select("id")
                  ->from("#__fields")
                  ->where("group_id=".$db->quote($fieldGroupId))
                  ->order("id ASC");

            $fieldCid = $db->setQuery($query)->loadColumn();

            if($fieldCid)
            {
                foreach($fieldCid as $fieldId)
                {
                    $query->clear()
                          ->insert($db->quoteName("#__fields_categories"))
                          ->columns($db->quoteName([
                                "field_id",
                                "category_id"
                          ]))
                          ->values(implode(',', [
                                $fieldId,
                                $new_row_id
                          ]));

                    $db->setQuery($query)->execute();
                }
            }
        }

        // update child if have (level, path)
        if(!empty($child))
        {
            $query->clear()
                  ->update("#__categories")
                  ->set([
                    "level=level+1",
                    "path=CONCAT(".$db->quote($cat->alias."/").", path)",
                    "parent_id=IF(parent_id>1,parent_id,".$new_row_id.")"
                  ])
                  ->where("id IN(".implode(",", $child).")");

            $db->setQuery($query)->execute();
        }

        // try find parent id
        if($cat->parent>0)
        {
            $child[]=$new_row_id;
            $this->importCategory($cat->parent, $child);
        }

        return $new_row_id;
    }

    protected function importFields($offset=0)
    {
        // import fields and fields groups
        $types = [
            "multipleSelect" => "checkboxes",
            "date" => "calendar",
            "link" => "url",
            "textfield" => "text",
            "image" => "media"
        ];

        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        $query->select("*")
              ->from("#__k2_extra_fields_groups")
              ->order("id ASC");

        $groups = $db->setQuery($query, $offset, 30)->loadObjectList();

        if(!$groups) return;

        $stt = 1;
        $time = time();

        foreach($groups as $group)
        {
            $query->clear()
                  ->insert($db->quoteName("#__fields_groups"))
                  ->columns($db->quoteName(["context", "title", "state", "language", "created", "created_by"]))
                  ->values(implode(',', [
                    $db->quote("com_content.article"),
                    $db->quote($group->name),
                    $db->quote(1),
                    $db->quote("*"),
                    $db->quote($date),
                    $db->quote($user->id)
                  ]));

            $db->setQuery($query)->execute();
            $new_row_id = $db->insertid();

            $this->fieldGroupMap[$group->id]=$new_row_id;

            // insert fields
            $query->clear()
                  ->select("*")
                  ->from("#__k2_extra_fields")
                  ->where($db->quoteName("group")."=".$db->quote($group->id))
                  ->order("id ASC");

            $fields = $db->setQuery($query)->loadObjectList();

            if(!$fields) continue;

            foreach($fields as $field)
            {
                $stt++;

                $info = json_decode($field->value);
                $name = $info[0]->alias;
                if(empty($name))
                    $name = $time+$stt;
                $fieldparams = '';
                if($field->type=='multipleSelect' || $field->type=='select')
                {
                    $fieldparams = ['options'=>[]];
                    $fieldIndex = 0;

                    foreach($info as $if)
                    {
                        $fieldparams['options']['options'.$fieldIndex]=['name'=>$if->name, 'value'=>$if->value];

                        $fieldIndex++;
                    }

                    $fieldparams = json_encode($fieldparams);
                }

                $query->clear()
                      ->insert($db->quoteName("#__fields"))
                      ->columns($db->quoteName([
                        "context",
                        "group_id",
                        "title",
                        "name",
                        "label",
                        "default_value",
                        "type",
                        "state",
                        "required",
                        "ordering",
                        "language",
                        "created_time",
                        "created_user_id",
                        "fieldparams"
                      ]))
                      ->values(implode(',', [
                        $db->quote("com_content.article"),
                        $db->quote($new_row_id),
                        $db->quote($field->name),
                        $db->quote($name),
                        $db->quote($field->name),
                        $db->quote(@$info[0]->value),
                        $db->quote(array_key_exists($field->type, $types)?$types[$field->type]:$field->type),
                        $db->quote($field->published),
                        $db->quote(@$info[0]->required),
                        $db->quote($field->ordering),
                        $db->quote("*"),
                        $db->quote($date),
                        $db->quote($user->id),
                        $db->quote($fieldparams)
                      ]));

                $db->setQuery($query)->execute();
                $new_row_idx = $db->insertid();

                $fieldType = array_key_exists($field->type, $types)?$types[$field->type]:$field->type;
                $this->fieldMap[$field->id]=$new_row_idx;
                $this->fieldMapType[$field->id]=$fieldType;
            }
        }

        $this->importFields($offset+30);
    }

    protected function importField($field, $aid)
    {
        if(!array_key_exists($field->id, $this->fieldMap)) return;

        if(is_array($field->value))
        {
            $type = $this->fieldMapType[$field->id];

            if($type=='url')
            {
                $field->value = $field->value[1];
            } else {
                foreach($field->value as $v)
                {
                    $clone = $field;
                    $clone->value = $v;

                    $this->importField($clone, $aid);
                }
                
                return;
            }
        }

        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        $query->insert($db->quoteName("#__fields_values"))
              ->columns($db->quoteName(["field_id", "item_id", "value"]))
              ->values(implode(',', [
                    $db->quote($this->fieldMap[$field->id]),
                    $db->quote($aid),
                    $db->quote($field->value)
              ]));

        $db->setQuery($query)->execute();
    }

    protected function importTags($nid, $oid, $core_id=0)
    {
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);
        $user = JFactory::getUser();
        $date = JFactory::getDate()->toSql();

        $query->select("t.*")
              ->from("#__k2_tags_xref xf")
              ->leftJoin("#__k2_tags t ON xf.tagID=t.id")
              ->where([
                    "itemID=".$db->quote($oid),
                    "t.id IS NOT NULL"
              ])
              ->group("t.id");

        $tags = $db->setQuery($query)->loadObjectList();

        if(!$tags) return;

        jimport('joomla.filter.output');
        $cid = [];

        foreach($tags as $tag)
        {
            $alias = JFilterOutput::stringURLSafe($tag->name);
            $query->clear()
                  ->select("id")
                  ->from("#__tags")
                  ->where("alias=".$db->quote($alias));

            $exist = $db->setQuery($query)->loadResult();

            if($exist)
            {
                $cid[]=$exist;
                continue;
            }

            // find max lft
            $query->clear()
                  ->select("MAX(lft)")
                  ->from("#__tags");
            $max = $db->setQuery($query)->loadResult();
            $max = !$max?0:(int)$max;

            $query->clear()
                  ->insert($db->quoteName("#__tags"))
                  ->columns($db->quoteName([
                        "parent_id",
                        "lft", "rgt",
                        "level",
                        "path",
                        "title",
                        "alias",
                        "published",
                        "access",
                        "created_user_id",
                        "created_time",
                        "language"
                  ]))
                  ->values(implode(',', [
                        1,
                        $max+2, $max+3,
                        1,
                        $db->quote($alias),
                        $db->quote($tag->name),
                        $db->quote($alias),
                        $tag->published,
                        1,
                        $db->quote($user->id),
                        $db->quote($date),
                        $db->quote("*")
                  ]));

            $db->setQuery($query)->execute();
            $cid[] = $db->insertid();
        }

        if(empty($cid)) return;

        foreach($cid as $id)
        {
            $query->clear()
                  ->delete('#__contentitem_tag_map')
                  ->where([
                    'type_id=1',
                    'content_item_id='.$nid,
                    'tag_id='.$id
                  ]);
            $db->setQuery($query)->execute();

            $query->clear()
                  ->insert($db->quoteName("#__contentitem_tag_map"))
                  ->columns($db->quoteName([
                        "type_alias",
                        "core_content_id",
                        "content_item_id",
                        "tag_id",
                        "tag_date",
                        "type_id"
                  ]))
                  ->values(implode(',', [
                        $db->quote("com_content.article"),
                        $core_id,
                        $nid,
                        $id,
                        $db->quote($date),
                        1
                  ]));

            $db->setQuery($query)->execute();
        }
    }

    protected function importFiles($files, $folder, $id, $a)
    {
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        foreach($files as $f)
        {
            $pathFile = JPATH_ROOT.'/media/k2/attachments/'.$f->filename;
            if(!file_exists($pathFile)) continue;

            $type = strtolower(JFile::getExt($f->filename));
            $filename = substr(md5($f->title . $a->created . $a->created_by), 0, 16) . '.' . $type;

            $data = [];
            $data['name'] = $f->title;
            $data['description'] = $f->titleAttribute;
            $data['content_id'] = $id;
            $data['created'] = $a->created;
            $data['created_by'] = $a->created_by;
            $data['fileSize'] = filesize($pathFile);
            $data['fileName'] = $filename;
            $data['fileType'] = in_array($type, ['png', 'jpg', 'jpeg', 'gif'])?'image':(in_array($type, ['zip', 'rar'])?'compression':'unknown');
            $data['downloadCount'] = (int) $f->hits;

            // copy file
            $dest = JPATH_ROOT.'/'.$folder.'/'.$filename;

            if(JFile::copy($pathFile, $dest))
            {
                $columns=[]; $values=[];
                foreach($data as $k=>$v)
                {
                    $columns[]=$k;
                    $values[]=$db->quote($v);
                }

                $query
                    ->clear()
                    ->insert($db->quoteName('#__jmpatm_files'))
                    ->columns($db->quoteName($columns))
                    ->values(implode(',', $values));

                $db->setQuery($query)->execute();
            }
        }
    }

    protected function toJson($data)
    {
        $app = JFactory::getApplication();
        header("Content-Type:application/json");
        echo json_encode($data);
        $app->close();
    }
}